import os
import time
import json
from selenium import webdriver
from selenium.webdriver.chrome.service import Service
from selenium.webdriver.chrome.options import Options
from selenium.webdriver.common.by import By
from selenium.webdriver.support.ui import WebDriverWait
from selenium.webdriver.support import expected_conditions as EC
from selenium.common.exceptions import TimeoutException
from webdriver_manager.chrome import ChromeDriverManager
import gspread
from oauth2client.service_account import ServiceAccountCredentials

# Sheet and credential info
GOOGLE_SHEET_NAME = "RedditAuth"
SERVICE_ACCOUNT_FILE = "client_secret.json"

def authorize_google_sheet():
    scope = ['https://spreadsheets.google.com/feeds', 'https://www.googleapis.com/auth/drive']
    creds = ServiceAccountCredentials.from_json_keyfile_name(SERVICE_ACCOUNT_FILE, scope)
    client = gspread.authorize(creds)
    return client

def get_sheet_data():
    client = authorize_google_sheet()
    sheet = client.open(GOOGLE_SHEET_NAME).sheet1
    data = sheet.get_all_records()
    return sheet, data

def list_profiles():
    profile_base_path = os.path.abspath("chrome-profiles")
    if not os.path.exists(profile_base_path):
        print("[!] 'chrome-profiles' folder does not exist.")
        return []

    folders = [f for f in os.listdir(profile_base_path)
               if os.path.isdir(os.path.join(profile_base_path, f)) and not f.endswith("_backup")]

    if not folders:
        print("[!] No profiles found in 'chrome-profiles'.")
        return []

    print("\n📁 Chrome Profiles Found:\n" + "-" * 30)
    for idx, folder in enumerate(folders, start=1):
        print(f"{idx}. {folder}")
    print("-" * 30)
    print(f"Total: {len(folders)} profiles\n")

    return folders

def create_proxy_extension(proxy_str):
    # proxy_str = "address:port:username:password"
    # Chrome requires an extension to handle authenticated proxy

    # Parse proxy details
    try:
        address, port, user, pwd = proxy_str.split(":")
    except Exception:
        print(f"[!] Proxy string format invalid: {proxy_str}")
        return None

    manifest_json = """
    {
        "version": "1.0.0",
        "manifest_version": 2,
        "name": "Chrome Proxy",
        "permissions": [
            "proxy",
            "tabs",
            "unlimitedStorage",
            "storage",
            "<all_urls>",
            "webRequest",
            "webRequestBlocking"
        ],
        "background": {
            "scripts": ["background.js"]
        },
        "minimum_chrome_version":"22.0.0"
    }
    """

    background_js = f"""
    var config = {{
            mode: "fixed_servers",
            rules: {{
              singleProxy: {{
                scheme: "http",
                host: "{address}",
                port: parseInt({port})
              }},
              bypassList: ["localhost"]
            }}
          }};

    chrome.proxy.settings.set({{value: config, scope: "regular"}}, function() {{}});

    function callbackFn(details) {{
        return {{
            authCredentials: {{
                username: "{user}",
                password: "{pwd}"
            }}
        }};
    }}

    chrome.webRequest.onAuthRequired.addListener(
                callbackFn,
                {{urls: ["<all_urls>"]}},
                ['blocking']
    );
    """

    # Write these to a temp directory and pack as extension
    import zipfile
    import tempfile

    temp_dir = tempfile.mkdtemp()
    manifest_path = os.path.join(temp_dir, "manifest.json")
    background_path = os.path.join(temp_dir, "background.js")

    with open(manifest_path, "w") as f:
        f.write(manifest_json.strip())

    with open(background_path, "w") as f:
        f.write(background_js.strip())

    zip_path = os.path.join(temp_dir, "proxy_auth_extension.zip")
    with zipfile.ZipFile(zip_path, 'w') as zp:
        zp.write(manifest_path, "manifest.json")
        zp.write(background_path, "background.js")

    return zip_path

def launch_reddit(account):
    profile_name = account['profile_name']
    proxy_str = account.get('proxy')
    profile_path = os.path.abspath(os.path.join("chrome-profiles", profile_name))

    chrome_options = Options()
    chrome_options.add_argument(f"--user-data-dir={profile_path}")
    chrome_options.add_argument("--profile-directory=Default")
    chrome_options.add_argument("--disable-blink-features=AutomationControlled")
    chrome_options.add_argument("--no-sandbox")
    chrome_options.add_argument("--start-maximized")
    chrome_options.add_experimental_option("excludeSwitches", ["enable-automation"])
    chrome_options.add_experimental_option('useAutomationExtension', False)

    if proxy_str:
        proxy_extension_path = create_proxy_extension(proxy_str)
        if proxy_extension_path:
            chrome_options.add_extension(proxy_extension_path)
        else:
            print(f"[!] Failed to create proxy extension for {profile_name}, skipping proxy.")

    try:
        driver = webdriver.Chrome(service=Service(ChromeDriverManager().install()), options=chrome_options)
        driver.get("https://www.reddit.com")

        WebDriverWait(driver, 15).until(EC.presence_of_element_located((By.TAG_NAME, "body")))

        # Check if logged in
        try:
            WebDriverWait(driver, 5).until(
                EC.presence_of_element_located((By.XPATH, "//a[contains(@href, '/submit')]"))
            )
            # If proxy used, show proxy part
            if proxy_str:
                proxy_address = proxy_str.split(":")[0]
                print(f"[✓] {profile_name} account is active under proxy {proxy_address}")
            else:
                print(f"[✓] {profile_name} account is actively stay in browser.")
        except TimeoutException:
            print(f"[!] {profile_name} is not logged in. Skipping login in this script.")

        # Optional: visit IP check page to confirm proxy IP (commented out)
        # driver.get("https://api.ipify.org?format=json")
        # ip_json = driver.find_element(By.TAG_NAME, "body").text
        # print(f"[Proxy IP Check] {profile_name} IP: {ip_json}")

        time.sleep(5)
        return driver   # <-- RETURN driver instance, do NOT quit here

    except Exception as e:
        print(f"[!] Failed to open {profile_name}: {str(e)}")
        return None

def warm_up_accounts():
    profiles = list_profiles()
    if not profiles:
        return

    user_input = input("How many account you want to warming up simulationly? (Ex: 1,2,3 or ALL): ").strip().lower()

    use_proxy = input("Do you want to use proxy? (y/n): ").strip().lower()
    use_proxy_flag = use_proxy == "y"

    sheet, rows = get_sheet_data()

    # Map profiles with username/password/proxy from sheet
    accounts = []
    for i, profile_name in enumerate(profiles):
        matched_row = None
        for row in rows:
            sheet_username = str(row.get('username', '')).lower().replace('@', '').strip()
            prof_norm = profile_name.lower().replace('@', '').strip()
            if sheet_username == prof_norm:
                matched_row = row
                break
        if not matched_row:
            matched_row = rows[i] if i < len(rows) else {}

        account = {
            "profile_name": profile_name,
            "username": matched_row.get("username", ""),
            "password": matched_row.get("password", ""),
            "proxy": matched_row.get("proxy", "") if use_proxy_flag else None,
        }
        accounts.append(account)

    if user_input == "all":
        batch_size = len(accounts)
    else:
        try:
            batch_size = max(1, min(10, int(user_input)))
        except:
            print("[!] Invalid input, using default value: 1")
            batch_size = 1

    total_batches = (len(accounts) + batch_size - 1) // batch_size

    for i in range(total_batches):
        batch = accounts[i * batch_size:(i + 1) * batch_size]
        print(f"\n🚀 Launching batch {i + 1}/{total_batches}...\n")

        drivers = []
        for account in batch:
            driver = launch_reddit(account)
            if driver:
                drivers.append(driver)

        input("Press ENTER then go next batch.")

        # Close all browsers after ENTER pressed
        for driver in drivers:
            try:
                driver.quit()
            except Exception:
                pass

    print("\n[✓] Warming up simulation completed.")

if __name__ == "__main__":
    warm_up_accounts()
