# command.py

import time
from selenium.webdriver.common.by import By
from selenium.webdriver.common.keys import Keys
from selenium.webdriver.support.ui import WebDriverWait
from selenium.webdriver.support import expected_conditions as EC

__all__ = ['click_profile', 'click_notification', 'search', 'search_feed_top', 
           'focus_on_tweet', 'like_tweet', 'comment_on_tweet']

def click_profile(driver, username):
    try:
        profile_link = WebDriverWait(driver, 5).until(
            EC.presence_of_element_located((By.CSS_SELECTOR, "a[data-testid='AppTabBar_Profile_Link']"))
        )
        profile_link.click()
        print("✅ Clicked profile link directly.")
    except Exception:
        print("⚠️ Profile link not found, using keyboard shortcut (g + p)")
        body = driver.find_element(By.TAG_NAME, "body")
        body.send_keys('g')
        time.sleep(0.2)
        body.send_keys('p')

    time.sleep(3)
    current_url = driver.current_url
    expected_url = f"https://x.com/{username.replace('@', '')}"
    if current_url.startswith(expected_url):
        print(f"✅ Profile URL OK: {current_url}")
    else:
        print(f"❌ Expected profile URL not matched. Got: {current_url}")

def click_notification(driver):
    body = driver.find_element(By.TAG_NAME, "body")
    for attempt in range(2):
        body.send_keys('g')
        time.sleep(0.2)
        body.send_keys('n')
        time.sleep(3)
        if "x.com/notifications" in driver.current_url:
            print("✅ Reached notifications page.")
            break
        else:
            print("❌ Notification URL not detected, retrying...")

def search(driver, keyword):
    print("🔍 Searching...")

    try:
        # Wait for the search input to be visible
        search_input = WebDriverWait(driver, 10).until(
            EC.element_to_be_clickable((By.CSS_SELECTOR, "input[data-testid='SearchBox_Search_Input']"))
        )
        search_input.click()
        search_input.clear()

        if keyword:
            print(f"✍ Typing keyword: {keyword}")
            search_input.send_keys(keyword)
            time.sleep(0.5)
            search_input.send_keys(Keys.ENTER)
            print("✅ Search submitted.")
        else:
            print("⚠️ No keyword provided for search command.")

    except Exception as e:
        print(f"[!] Could not interact with search input: {str(e)}")


def search_feed_top(driver, keyword, state_no=0):
    print("🔎 Running 'search_feed_top'...")
    search(driver, keyword)
    
    # Wait for search results to load
    time.sleep(3)
    
    try:
        # Click on "Top" tab
        top_tab = WebDriverWait(driver, 10).until(
            EC.element_to_be_clickable((By.XPATH, "//span[contains(text(), 'Top')]/ancestor::a"))
        )
        top_tab.click()
        print("✅ Clicked on Top tab")
        
        # If state_no is provided, focus on that tweet
        if state_no > 0:
            focus_on_tweet(driver, state_no)
            
    except Exception as e:
        print(f"[!] Could not find Top tab: {str(e)}")

def focus_on_tweet(driver, state_no):
    print(f"🔍 Focusing on tweet number {state_no}...")
    
    # Wait for page to load
    time.sleep(3)
    
    # Get all tweet elements that have a view count
    tweet_count = 0
    attempts = 0
    max_attempts = state_no * 2  # Prevent infinite loop
    
    while tweet_count < state_no and attempts < max_attempts:
        attempts += 1
        
        # Press 'j' to move to next tweet
        body = driver.find_element(By.TAG_NAME, "body")
        body.send_keys('j')
        time.sleep(0.5)
        
        try:
            # Check if current focused element has view count (indicates it's a tweet)
            WebDriverWait(driver, 1).until(
                EC.presence_of_element_located((By.XPATH, "//*[contains(@aria-label, 'views')]")))
            tweet_count += 1
            print(f"➡️ Moved to tweet {tweet_count}/{state_no}")
            
            # Store the currently focused tweet element
            focused_tweet = driver.switch_to.active_element
        except:
            # Not a tweet element, continue
            pass
    
    if tweet_count >= state_no:
        print(f"✅ Successfully focused on tweet {state_no}")
        try:
            # Get the tweet text for verification
            tweet_text = WebDriverWait(driver, 3).until(
                EC.presence_of_element_located((By.CSS_SELECTOR, "[data-testid='tweetText']")))
            print(f"📝 Tweet content: {tweet_text.text[:50]}...")
            return focused_tweet
        except:
            print("ℹ️ Could not retrieve tweet text")
            return None
    else:
        print(f"⚠️ Could not find tweet {state_no} after {max_attempts} attempts")
        return None

def like_tweet(driver, tweet_element, state_no):
    try:
        # Wait for the like button inside the tweet_element to be clickable
        like_button = WebDriverWait(tweet_element, 10).until(
            EC.element_to_be_clickable((By.CSS_SELECTOR, 'button[data-testid="like"]'))
        )
        like_button.click()
        print(f"👍 Liked tweet #{state_no}")
    except Exception as e:
        print(f"[!] Failed to like tweet #{state_no}: {e}")

def comment_on_tweet(driver, comment_text, tweet_element):
    try:
        # Click the reply button inside the tweet element
        reply_button = WebDriverWait(tweet_element, 10).until(
            EC.element_to_be_clickable((By.CSS_SELECTOR, 'button[data-testid="reply"]'))
        )
        reply_button.click()
        
        # Wait for the comment input box to appear
        comment_input = WebDriverWait(driver, 10).until(
            EC.presence_of_element_located((By.CSS_SELECTOR, 'div[role="textbox"][data-testid="tweetTextarea_0"]'))
        )
        
        # Clear and enter the comment text
        comment_input.clear()
        comment_input.send_keys(comment_text)
        
        # Submit comment with Ctrl+Enter
        comment_input.send_keys(Keys.CONTROL, Keys.ENTER)
        
        print(f"💬 Commented: {comment_text}")
        
    except Exception as e:
        print(f"[!] Failed to comment: {e}")