# warming_up.py

import os
import time
import gspread

from selenium import webdriver
from selenium.webdriver.chrome.service import Service
from selenium.webdriver.chrome.options import Options
from selenium.webdriver.common.by import By
from selenium.webdriver.support.ui import WebDriverWait
from selenium.webdriver.support import expected_conditions as EC
from selenium.common.exceptions import TimeoutException
from webdriver_manager.chrome import ChromeDriverManager
from oauth2client.service_account import ServiceAccountCredentials
from concurrent.futures import ThreadPoolExecutor
from command import process_command  # 🟢 single unified call

# ────────────────────────────────────────────────
# GOOGLE SHEET SETUP
# ────────────────────────────────────────────────
GOOGLE_SHEET_NAME = "RedditAuth"
SERVICE_ACCOUNT_FILE = "client_secret.json"

def authorize_google_sheet():
    scope = ['https://spreadsheets.google.com/feeds', 'https://www.googleapis.com/auth/drive']
    creds = ServiceAccountCredentials.from_json_keyfile_name(SERVICE_ACCOUNT_FILE, scope)
    client = gspread.authorize(creds)
    return client

def get_sheet_data():
    client = authorize_google_sheet()
    sheet = client.open(GOOGLE_SHEET_NAME)
    sheet1 = sheet.worksheet("Sheet1")
    command_sheet = sheet.worksheet("command")
    accounts = sheet1.get_all_records()
    commands = command_sheet.get_all_records()
    return sheet1, accounts, commands

# ────────────────────────────────────────────────
# CHROME PROFILE MANAGEMENT
# ────────────────────────────────────────────────
def list_profiles():
    profile_base_path = os.path.abspath("chrome-profiles")
    if not os.path.exists(profile_base_path):
        print("[!] 'chrome-profiles' folder does not exist.")
        return []

    folders = [f for f in os.listdir(profile_base_path)
               if os.path.isdir(os.path.join(profile_base_path, f)) and not f.endswith("_backup")]

    if not folders:
        print("[!] No profiles found in 'chrome-profiles'.")
        return []

    print("\n📁 Chrome Profiles Found:\n" + "-" * 30)
    for idx, folder in enumerate(folders, start=1):
        print(f"{idx}. {folder}")
    print("-" * 30)
    print(f"Total: {len(folders)} profiles\n")

    return folders

# ────────────────────────────────────────────────
# REDDIT LAUNCHER
# ────────────────────────────────────────────────
def launch_reddit(account):
    profile_name = account['profile_name']
    profile_path = os.path.abspath(os.path.join("chrome-profiles", profile_name))

    chrome_options = Options()
    chrome_options.add_argument(f"--user-data-dir={profile_path}")
    chrome_options.add_argument("--profile-directory=Default")
    chrome_options.add_argument("--disable-blink-features=AutomationControlled")
    chrome_options.add_argument("--no-sandbox")
    chrome_options.add_argument("--start-maximized")
    chrome_options.add_experimental_option("excludeSwitches", ["enable-automation"])
    chrome_options.add_experimental_option('useAutomationExtension', False)

    try:
        driver = webdriver.Chrome(service=Service(ChromeDriverManager().install()), options=chrome_options)
        driver.get("https://www.reddit.com")
        WebDriverWait(driver, 15).until(EC.presence_of_element_located((By.TAG_NAME, "body")))

        try:
            WebDriverWait(driver, 5).until(
                EC.presence_of_element_located((By.XPATH, "//a[contains(@href, '/submit')]"))
            )
            print(f"[✓] {profile_name} is logged in.")
        except TimeoutException:
            print(f"[!] {profile_name} is not logged in. Skipping further actions.")
            return driver

        time.sleep(3)
        return driver

    except Exception as e:
        print(f"[!] Failed to open {profile_name}: {str(e)}")
        return None

# ────────────────────────────────────────────────
# MAIN RUNNER
# ────────────────────────────────────────────────
def warm_up_accounts():
    profiles = list_profiles()
    if not profiles:
        return

    user_input = input("How many accounts do you want to warm up simultaneously? (Ex: 1,2,3 or ALL): ").strip().lower()
    use_proxy = input("Do you want to use proxy from Webshare? (y/n): ").strip().lower()
    if use_proxy == 'y':
        print("Thanks for choosing y, but proxy function is not set up yet. Under construction.")

    sheet, rows, command_data = get_sheet_data()

    accounts = []
    for i, profile_name in enumerate(profiles):
        matched_row = rows[i] if i < len(rows) else {}
        account = {
            "profile_name": profile_name,
            "username": matched_row.get("username", ""),
            "password": matched_row.get("password", ""),
            "proxy": None,
        }
        accounts.append(account)

    if user_input == "all":
        batch_size = len(accounts)
    else:
        try:
            batch_size = max(1, min(10, int(user_input)))
        except:
            print("[!] Invalid input, using default value: 1")
            batch_size = 1

    total_batches = (len(accounts) + batch_size - 1) // batch_size

    for i in range(total_batches):
        batch = accounts[i * batch_size:(i + 1) * batch_size]
        print(f"\n🚀 Launching batch {i + 1}/{total_batches}...\n")

        drivers = []
        with ThreadPoolExecutor(max_workers=batch_size) as executor:
            drivers = list(executor.map(launch_reddit, batch))

        for account, driver in zip(batch, drivers):
            if driver:
                for cmd_row in command_data:
                    process_command(driver, account, cmd_row)

        input("Press ENTER to continue to the next batch...")

        for driver in drivers:
            try:
                if driver:
                    driver.quit()
            except Exception:
                pass

    print("\n[✓] Warming up simulation completed.")

if __name__ == "__main__":
    warm_up_accounts()
