import os
import time
import shutil
import gspread
from selenium import webdriver
from selenium.webdriver.chrome.service import Service
from selenium.webdriver.chrome.options import Options
from selenium.webdriver.common.by import By
from selenium.webdriver.common.keys import Keys
from selenium.webdriver.support.ui import WebDriverWait
from selenium.webdriver.support import expected_conditions as EC
from webdriver_manager.chrome import ChromeDriverManager
from multiprocessing import Pool
from oauth2client.service_account import ServiceAccountCredentials

# Sheet and credential info
GOOGLE_SHEET_NAME = "RedditAuth"  # Change the sheet name here
SERVICE_ACCOUNT_FILE = "client_secret.json"

def authorize_google_sheet():
    scope = ['https://spreadsheets.google.com/feeds', 'https://www.googleapis.com/auth/drive']
    creds = ServiceAccountCredentials.from_json_keyfile_name(SERVICE_ACCOUNT_FILE, scope)
    client = gspread.authorize(creds)
    return client

def get_sheet_data():
    client = authorize_google_sheet()
    sheet = client.open(GOOGLE_SHEET_NAME).sheet1
    data = sheet.get_all_records()
    return sheet, data

def update_status(sheet, row, status):
    sheet.update_cell(row + 2, 3, status)

def show_auth_menu():
    while True:
        print("\n" + "-" * 30)
        print(" Reddit Authorization Menu ".center(30))
        print("-" * 30)
        print("1. Process accounts from Google Sheet")
        print("2. Single account (manual input)")
        print("3. Check profiles from sheet status")
        print("4. Back to main menu")

        try:
            choice = input("\nChoose (1-4): ").strip()
            if choice == "1":
                batch_size = input("\nHow many windows to run simultaneously? (1-10 recommended): ").strip()
                try:
                    batch_size = max(1, min(10, int(batch_size)))
                    process_sheet_accounts(batch_size)
                except ValueError:
                    print("! Invalid number, using default 1")
                    process_sheet_accounts(1)
            elif choice == "2":
                manual_account_login()
            elif choice == "3":
                check_profiles_with_status()
            elif choice == "4":
                return
            else:
                print("\n[!] Invalid choice")
        except Exception as e:
            print(f"\n[!] Error: {str(e)}")

def manual_account_login():
    username = input("\nEnter Reddit username: ").strip()
    password = input("Enter Reddit password: ").strip()

    if not username or not password:
        print("\n[!] Username and password are required")
        return

    print(f"\nProcessing: {username}")
    if reddit_login_single(username, password):
        print(f"[✓] Successfully logged in: {username}")
    else:
        print(f"[!] Failed to log in: {username}")

def process_sheet_accounts(batch_size=1):
    try:
        sheet, rows = get_sheet_data()  # Fetch data from RedditAuth sheet

        pending_accounts = []
        for idx, row in enumerate(rows):
            username = row.get('username')
            password = row.get('password')
            status = str(row.get('status', '')).lower()

            if username and password and status != 'completed':
                pending_accounts.append({
                    'index': idx,
                    'username': username,
                    'password': password,
                })

        if not pending_accounts:
            print("\n[!] No pending accounts found.")
            return

        total_batches = (len(pending_accounts) // batch_size) + (1 if len(pending_accounts) % batch_size else 0)

        for i in range(total_batches):
            batch = pending_accounts[i * batch_size:(i + 1) * batch_size]
            print(f"\nStarting batch {i + 1} with {len(batch)} accounts")
            with Pool(batch_size) as pool:
                results = pool.starmap(process_account_in_window, [(account,) for account in batch])

            for account, success in zip(batch, results):
                if success:
                    update_status(sheet, account['index'], 'completed')
                    print(f"[✓] Marked completed: {account['username']}")

        print("\n[✓] All batches processed.")

    except Exception as e:
        print(f"\n[!] Error: {str(e)}")

def process_account_in_window(account):
    return reddit_login_single(account['username'], account['password'])

def reddit_login_single(username, password):
    chrome_options = Options()
    profile_name = username.replace('@', '_')
    profile_path = os.path.abspath(os.path.join("chrome-profiles", profile_name))
    os.makedirs(profile_path, exist_ok=True)

    chrome_options.add_argument(f"--user-data-dir={profile_path}")
    chrome_options.add_argument("--profile-directory=Default")
    chrome_options.add_argument("--disable-blink-features=AutomationControlled")
    chrome_options.add_argument("--no-sandbox")
    chrome_options.add_argument("--start-maximized")
    chrome_options.add_experimental_option("excludeSwitches", ["enable-automation"])
    chrome_options.add_experimental_option('useAutomationExtension', False)
    chrome_options.add_argument("--disable-extensions")
    chrome_options.add_argument("--disable-default-apps")
    chrome_options.add_argument("--disable-infobars")
    chrome_options.add_argument("--disable-notifications")

    try:
        driver = webdriver.Chrome(service=Service(ChromeDriverManager().install()), options=chrome_options)

        print(f"\n=== Processing: {username} ===")
        driver.get("https://www.reddit.com/login/")
        time.sleep(3)

        # Enter username
        user_input = WebDriverWait(driver, 15).until(EC.visibility_of_element_located((By.NAME, "username")))
        user_input.send_keys(username)

        # Enter password
        pwd_input = WebDriverWait(driver, 15).until(EC.visibility_of_element_located((By.NAME, "password")))
        pwd_input.send_keys(password)

        # Wait and press ENTER to log in
        time.sleep(2)
        pwd_input.send_keys(Keys.ENTER)

        # Wait for login to complete
        time.sleep(5)

        # Check if redirected and logged in
        try:
            WebDriverWait(driver, 10).until(EC.presence_of_element_located((By.XPATH, "//a[contains(@href, '/submit')]")))
            print("[✓] Successfully logged in.")
        except:
            print("[!] Login failed or not redirected to home page.")
            driver.quit()
            return False

        # Backup the profile if login is successful
        backup_path = os.path.join("chrome-profiles", f"{profile_name}_backup")
        try:
            if os.path.exists(backup_path):
                shutil.rmtree(backup_path)
            shutil.copytree(profile_path, backup_path)
            print(f"[✓] Profile saved for {username}")
        except Exception as e:
            print(f"[!] Backup failed: {str(e)}")

        driver.quit()
        return True

    except Exception as e:
        print(f"[!] Browser error: {str(e)}")
        return False

def check_profiles_with_status():
    try:
        sheet, rows = get_sheet_data()
        completed = [row for row in rows if str(row.get('status', '')).lower() == 'completed']

        if not completed:
            print("\n[!] No completed profiles found.")
            return

        for row in completed:
            username = row['username']
            profile_name = username.replace('@', '_')
            profile_path = os.path.join("chrome-profiles", profile_name)

            if os.path.exists(profile_path):
                print(f"[✓] Profile exists: {username}")
            else:
                print(f"[!] Profile missing: {username}")

        print("\n[✓] Profile check complete.")

    except Exception as e:
        print(f"[!] Error checking profiles: {str(e)}")

if __name__ == "__main__":
    show_auth_menu()
